// rooms.js - Room management functionality
import { initializeApp } from "https://www.gstatic.com/firebasejs/9.22.0/firebase-app.js";
import {
  getDatabase,
  ref,
  push,
  set,
  onValue,
  get,
  query,
  orderByChild,
  equalTo,
  serverTimestamp,
  update,
  remove,
  runTransaction,
  onDisconnect,
} from "https://www.gstatic.com/firebasejs/9.22.0/firebase-database.js";

document.addEventListener("DOMContentLoaded", () => {
  // Initialize Firebase using the config from window.APP_CONFIG
  const firebaseConfig = {
    apiKey: window.APP_CONFIG.FIREBASE_API_KEY,
    authDomain: window.APP_CONFIG.FIREBASE_AUTH_DOMAIN,
    projectId: window.APP_CONFIG.FIREBASE_PROJECT_ID,
    storageBucket: window.APP_CONFIG.FIREBASE_STORAGE_BUCKET,
    messagingSenderId: window.APP_CONFIG.FIREBASE_MESSAGING_SENDER_ID,
    appId: window.APP_CONFIG.FIREBASE_APP_ID,
    measurementId: window.APP_CONFIG.FIREBASE_MEASUREMENT_ID,
    databaseURL: window.APP_CONFIG.FIREBASE_DATABASE_URL,
  };

  // Initialize Firebase
  try {
    const app = initializeApp(firebaseConfig);
    const db = getDatabase(app);

    // DOM elements
    const roomsList = document.getElementById("rooms-list");
    const roomNameInput = document.getElementById("room-name-input");
    const roomTypeSelect = document.getElementById("room-type-select");
    const createRoomButton = document.getElementById("create-room-button");
    const refreshRoomsButton = document.getElementById("refresh-rooms-button");
    const roomFilter = document.getElementById("room-filter");
    const privateRoomDialog = document.getElementById("private-room-dialog");
    const roomPasswordInput = document.getElementById("room-password-input");
    const confirmJoinButton = document.getElementById("confirm-join-button");
    const cancelJoinButton = document.getElementById("cancel-join-button");

    // Current state
    let currentRoomId = null;
    let currentUserId = generateUserId();
    let selectedPrivateRoomId = null;

    // Reference to rooms in Firebase
    const roomsRef = ref(db, "rooms");

    // Load rooms on initial load
    loadRooms();

    // Event listeners
    createRoomButton.addEventListener("click", createRoom);
    refreshRoomsButton.addEventListener("click", loadRooms);
    roomFilter.addEventListener("change", loadRooms);
    confirmJoinButton.addEventListener("click", confirmJoinPrivateRoom);
    cancelJoinButton.addEventListener("click", cancelJoinPrivateRoom);

    // Add a "Global Chat" option
    addGlobalChatOption();

    // Function to add a Global Chat option to the rooms list
    function addGlobalChatOption() {
      const globalChatElement = document.createElement("div");
      globalChatElement.className = "room-item";
      globalChatElement.id = "global-chat-room";

      if (currentRoomId === null) {
        globalChatElement.classList.add("active");
      }

      globalChatElement.innerHTML = `
        <div>
          <strong>Global Chat</strong>
          <div>
            <small>🌍 Global</small>
          </div>
        </div>
        <button class="join-button">Join</button>
      `;

      const joinButton = globalChatElement.querySelector(".join-button");
      joinButton.addEventListener("click", () => {
        joinGlobalChat();
      });

      // Insert at the beginning of the rooms list
      if (roomsList.firstChild) {
        roomsList.insertBefore(globalChatElement, roomsList.firstChild);
      } else {
        roomsList.appendChild(globalChatElement);
      }
    }

    // Function to join the global chat
    function joinGlobalChat() {
      // Clear current room ID to indicate we're in global chat
      currentRoomId = null;
      window.ChatApp.currentRoomId = null;

      // Update UI to show active room
      const roomItems = document.querySelectorAll(".room-item");
      roomItems.forEach((item) => {
        item.classList.remove("active");
      });

      // Highlight the global chat
      const globalChatElement = document.getElementById("global-chat-room");
      if (globalChatElement) {
        globalChatElement.classList.add("active");
      }

      // Load global messages
      const messagesRef = ref(db, "messages");
      const messagesQuery = query(messagesRef, orderByChild("timestamp"));

      // Clear previous message listeners
      if (window.currentMessageListener) {
        window.currentMessageListener();
      }

      // Set up new message listener for global chat
      window.currentMessageListener = onValue(messagesQuery, (snapshot) => {
        const messagesContainer = document.getElementById("messages-container");
        if (!messagesContainer) return;

        messagesContainer.innerHTML = "";

        if (!snapshot.exists()) {
          messagesContainer.innerHTML =
            "<p class='no-messages'>No messages in the global chat yet. Be the first to say something!</p>";
          return;
        }

        // Display messages
        const messages = [];
        snapshot.forEach((childSnapshot) => {
          const message = childSnapshot.val();
          message.key = childSnapshot.key;
          messages.push(message);
        });

        // Sort messages by timestamp
        messages.sort((a, b) => {
          return (a.timestamp || 0) - (b.timestamp || 0);
        });

        // Display messages
        messages.forEach((message) => {
          const messageElement = document.createElement("div");
          messageElement.className = "message";
          messageElement.innerHTML = `
            <span class="username">${message.username || "Anonymous"}:</span>
            <span class="text">${message.text}</span>
          `;
          messagesContainer.appendChild(messageElement);
        });

        // Scroll to bottom
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
      });

      // Update chat title
      const chatTitle = document.querySelector("#chat-container h2");
      if (chatTitle) {
        chatTitle.textContent = "Global Chat";
      }
    }

    // Function to load global messages
    function loadGlobalMessages() {
      const messagesRef = ref(db, "messages");
      const messagesQuery = query(
        messagesRef,
        orderByChild("timestamp"),
        limitToLast(50)
      );

      onValue(messagesQuery, (snapshot) => {
        const messagesContainer = document.getElementById("messages-container");
        if (!messagesContainer) return;

        messagesContainer.innerHTML = "";

        if (!snapshot.exists()) {
          messagesContainer.innerHTML =
            "<p class='no-messages'>No messages yet. Be the first to say something!</p>";
          return;
        }

        const messages = [];
        snapshot.forEach((childSnapshot) => {
          const message = childSnapshot.val();
          message.key = childSnapshot.key;
          messages.push(message);
        });

        // Sort messages by timestamp
        messages.sort((a, b) => {
          return (a.timestamp || 0) - (b.timestamp || 0);
        });

        // Display messages
        messages.forEach((message) => {
          const messageElement = document.createElement("div");
          messageElement.className = "message";
          messageElement.innerHTML = `
            <span class="username">${message.username || "Anonymous"}:</span>
            <span class="text">${message.text}</span>
          `;
          messagesContainer.appendChild(messageElement);
        });

        // Scroll to bottom
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
      });
    }

    // Function to create a new room
    // Function to create a new room
    async function createRoom() {
      const roomName = roomNameInput.value.trim();
      const roomType = roomTypeSelect.value;

      if (!roomName) {
        alert("Please enter a room name");
        return;
      }

      // For private rooms, show password dialog
      if (roomType === "private") {
        showCreatePrivateRoomDialog(roomName);
        return;
      }

      // Create public room directly
      createRoomWithDetails(roomName, roomType);
    }

    // Function to show dialog for creating private rooms
    function showCreatePrivateRoomDialog(roomName) {
      const dialog = document.getElementById("create-private-room-dialog");
      const passwordInput = document.getElementById(
        "create-room-password-input"
      );

      // Store the room name for later use
      dialog.dataset.roomName = roomName;

      // Show dialog and focus on password input
      dialog.style.display = "flex";
      passwordInput.value = "";
      passwordInput.focus();
    }

    const createPrivateRoomDialog = document.getElementById(
      "create-private-room-dialog"
    );
    const createRoomPasswordInput = document.getElementById(
      "create-room-password-input"
    );
    const confirmCreateButton = document.getElementById(
      "confirm-create-button"
    );
    const cancelCreateButton = document.getElementById("cancel-create-button");

    // Event listeners for create private room dialog
    confirmCreateButton.addEventListener("click", confirmCreatePrivateRoom);
    cancelCreateButton.addEventListener("click", cancelCreatePrivateRoom);

    // Handle Enter key in create password input
    createRoomPasswordInput.addEventListener("keydown", function (e) {
      if (e.key === "Enter") {
        e.preventDefault();
        confirmCreatePrivateRoom();
      }
    });

    // Function to confirm creating a private room
    function confirmCreatePrivateRoom() {
      const password = createRoomPasswordInput.value.trim();
      const roomName = createPrivateRoomDialog.dataset.roomName;

      if (!password) {
        // Instead of alert, show inline error message
        const errorMsg = document.createElement("p");
        errorMsg.className = "dialog-error";
        errorMsg.textContent = "Private rooms must have a password";

        // Remove any existing error message
        const existingError =
          createPrivateRoomDialog.querySelector(".dialog-error");
        if (existingError) {
          existingError.remove();
        }

        // Add error message to dialog
        const dialogContent =
          createPrivateRoomDialog.querySelector(".dialog-content");
        dialogContent.insertBefore(
          errorMsg,
          document.getElementById("confirm-create-button").parentNode
        );
        return;
      }

      // Create the room with the provided password
      createRoomWithDetails(roomName, "private", password);

      // Hide dialog
      createPrivateRoomDialog.style.display = "none";
    }

    // Function to cancel creating a private room
    function cancelCreatePrivateRoom() {
      createPrivateRoomDialog.style.display = "none";
    }

    // Function to handle the creation of a room with all details
    async function createRoomWithDetails(roomName, roomType, password = null) {
      // Create the room in Firebase
      const newRoomRef = push(roomsRef);
      const roomData = {
        name: roomName,
        type: roomType,
        createdBy: currentUserId,
        createdAt: serverTimestamp(),
        memberCount: 1,
        members: {
          [currentUserId]: true,
        },
      };

      // Add password for private rooms
      if (roomType === "private" && password) {
        roomData.password = password;
      }

      try {
        await set(newRoomRef, roomData);
        roomNameInput.value = "";
        loadRooms();

        // Switch to the new room
        joinRoom(newRoomRef.key, roomType === "private" ? password : null);
      } catch (error) {
        console.error("Error creating room:", error);
        alert("Failed to create room. Please try again.");
      }
    }

    // Function to load rooms based on filter
    function loadRooms() {
      const filter = roomFilter.value;
      let roomsQuery;

      switch (filter) {
        case "public":
          roomsQuery = query(roomsRef, orderByChild("type"), equalTo("public"));
          break;
        case "private":
          roomsQuery = query(
            roomsRef,
            orderByChild("type"),
            equalTo("private")
          );
          break;
        case "my":
          // This is a client-side filter since we can't query nested objects easily
          roomsQuery = roomsRef;
          break;
        default:
          roomsQuery = roomsRef;
      }

      onValue(roomsQuery, (snapshot) => {
        // Clear the rooms list but keep the global chat option
        const globalChatElement = document.getElementById("global-chat-room");
        roomsList.innerHTML = "";

        // Re-add the global chat option
        if (globalChatElement) {
          roomsList.appendChild(globalChatElement);
        } else {
          addGlobalChatOption();
        }

        if (!snapshot.exists()) {
          const noRoomsElement = document.createElement("p");
          noRoomsElement.textContent = "No rooms available";
          roomsList.appendChild(noRoomsElement);
          return;
        }

        const rooms = [];
        snapshot.forEach((childSnapshot) => {
          const room = childSnapshot.val();
          room.id = childSnapshot.key;

          // Filter "my" rooms client-side
          if (
            filter === "my" &&
            (!room.members || !room.members[currentUserId])
          ) {
            return;
          }

          rooms.push(room);
        });

        // Sort rooms by creation time (newest first)
        rooms.sort((a, b) => {
          return (b.createdAt || 0) - (a.createdAt || 0);
        });

        // Display rooms
        rooms.forEach((room) => {
          const roomElement = document.createElement("div");
          roomElement.className = "room-item";
          if (room.id === currentRoomId) {
            roomElement.classList.add("active");
          }

          const actualMemberCount = room.members
            ? Object.keys(room.members).length
            : 1;

          roomElement.innerHTML = `
            <div>
              <strong>${room.name}</strong>
              <div>
                <small>${
                  room.type === "private" ? "🔒 Private" : "🌐 Public"
                }</small>
                <small>👥 ${actualMemberCount}</small>
              </div>
            </div>
            <button class="join-button" data-room-id="${room.id}">Join</button>
          `;

          const joinButton = roomElement.querySelector(".join-button");
          joinButton.addEventListener("click", () => {
            // If it's a private room, show password dialog
            if (room.type === "private") {
              showPrivateRoomDialog(room.id);
            } else {
              joinRoom(room.id);
            }
          });

          roomsList.appendChild(roomElement);
        });

        // Highlight the active room
        if (currentRoomId === null) {
          const globalChatElement = document.getElementById("global-chat-room");
          if (globalChatElement) {
            globalChatElement.classList.add("active");
          }
        } else {
          const roomItems = document.querySelectorAll(".room-item");
          roomItems.forEach((item) => {
            const joinButton = item.querySelector(".join-button");
            if (joinButton && joinButton.dataset.roomId === currentRoomId) {
              item.classList.add("active");
            }
          });
        }
      });
    }

    // Function to show the private room password dialog
    function showPrivateRoomDialog(roomId) {
      selectedPrivateRoomId = roomId;
      privateRoomDialog.style.display = "flex";
      roomPasswordInput.value = "";
      roomPasswordInput.focus();
    }

    // Function to confirm joining a private room with password
    async function confirmJoinPrivateRoom() {
      const password = roomPasswordInput.value.trim();

      if (!password) {
        // Show inline error instead of alert
        showDialogError(privateRoomDialog, "Please enter the room password");
        return;
      }

      try {
        // Get the room data to verify password
        const roomSnapshot = await get(
          ref(db, `rooms/${selectedPrivateRoomId}`)
        );

        if (!roomSnapshot.exists()) {
          showDialogError(privateRoomDialog, "Room not found");
          return;
        }

        const roomData = roomSnapshot.val();

        // Check if password matches
        if (roomData.password !== password) {
          showDialogError(privateRoomDialog, "Incorrect password");
          return;
        }

        // Password is correct, join the room
        joinRoom(selectedPrivateRoomId, password);
        cancelJoinPrivateRoom(); // Close the dialog
      } catch (error) {
        console.error("Error joining private room:", error);
        showDialogError(
          privateRoomDialog,
          "Failed to join room. Please try again."
        );
      }
    }

    // Helper function to show dialog errors
    function showDialogError(dialogElement, errorMessage) {
      // Remove any existing error message
      const existingError = dialogElement.querySelector(".dialog-error");
      if (existingError) {
        existingError.remove();
      }

      // Create and add new error message
      const errorMsg = document.createElement("p");
      errorMsg.className = "dialog-error";
      errorMsg.textContent = errorMessage;

      const dialogContent = dialogElement.querySelector(".dialog-content");
      dialogContent.insertBefore(
        errorMsg,
        dialogContent.querySelector(".dialog-buttons")
      );
    }

    // Function to cancel joining a private room
    function cancelJoinPrivateRoom() {
      privateRoomDialog.style.display = "none";
      selectedPrivateRoomId = null;
    }

    // Function to join a room
    function joinRoom(roomId, password = null) {
      if (window.currentDisconnectHandler) {
        window.currentDisconnectHandler();
        window.currentDisconnectHandler = null;
      }
      // Get the room data
      const roomRef = ref(db, `rooms/${roomId}`);
      get(roomRef)
        .then((snapshot) => {
          if (!snapshot.exists()) {
            alert("Room not found!");
            return;
          }

          const roomData = snapshot.val();

          // Check if it's a private room and verify password
          if (roomData.type === "private") {
            if (password === null) {
              // Show password dialog
              selectedPrivateRoomId = roomId;
              privateRoomDialog.style.display = "flex";
              roomPasswordInput.value = "";
              roomPasswordInput.focus();
              return;
            } else if (password !== roomData.password) {
              alert("Incorrect password!");
              return;
            }
          }

          // Set the current room ID
          currentRoomId = roomId;
          window.ChatApp.currentRoomId = roomId;

          // Load messages for this specific room
          const roomMessagesRef = ref(db, `room-messages/${roomId}`);
          const messagesQuery = query(
            roomMessagesRef,
            orderByChild("timestamp")
          );

          // Clear previous message listeners
          if (window.currentMessageListener) {
            window.currentMessageListener();
          }

          // Set up new message listener for this room
          window.currentMessageListener = onValue(messagesQuery, (snapshot) => {
            const messagesContainer =
              document.getElementById("messages-container");
            if (!messagesContainer) return;

            messagesContainer.innerHTML = "";

            if (!snapshot.exists()) {
              messagesContainer.innerHTML =
                "<p class='no-messages'>No messages in this room yet. Be the first to say something!</p>";
              return;
            }

            // Display messages
            const messages = [];
            snapshot.forEach((childSnapshot) => {
              const message = childSnapshot.val();
              message.key = childSnapshot.key;
              messages.push(message);
            });

            // Sort messages by timestamp
            messages.sort((a, b) => {
              return (a.timestamp || 0) - (b.timestamp || 0);
            });

            // Display messages
            messages.forEach((message) => {
              const messageElement = document.createElement("div");
              messageElement.className = "message";
              messageElement.innerHTML = `
              <span class="username">${message.username || "Anonymous"}:</span>
              <span class="text">${message.text}</span>
            `;
              messagesContainer.appendChild(messageElement);
            });

            // Scroll to bottom
            messagesContainer.scrollTop = messagesContainer.scrollHeight;
          });

          // Update chat title
          const chatTitle = document.querySelector("#chat-container h2");
          if (chatTitle) {
            chatTitle.textContent = `Chat: ${roomData.name}`;
          }

          // Update UI to show active room
          const roomItems = document.querySelectorAll(".room-item");
          roomItems.forEach((item) => {
            item.classList.remove("active");
          });

          // Highlight the current room
          const currentRoomElement = document.getElementById(`room-${roomId}`);
          if (currentRoomElement) {
            currentRoomElement.classList.add("active");
          }

          // Update room participant count
          const participantsRef = ref(db, `room-participants/${roomId}`);
          const participantsQuery = query(
            participantsRef,
            orderByChild("userId"),
            equalTo(currentUserId)
          );

          get(participantsQuery)
            .then((snapshot) => {
              let newParticipantRef;

              // If user is not already in participants, add them
              if (!snapshot.exists()) {
                newParticipantRef = push(participantsRef);
                set(newParticipantRef, {
                  userId: currentUserId,
                  joinedAt: serverTimestamp(),
                });

                // Only update memberCount if this is a new participant
                const roomDataRef = ref(db, `rooms/${roomId}`);
                runTransaction(roomDataRef, (currentData) => {
                  if (currentData === null) return null;

                  // Initialize members object if it doesn't exist
                  if (!currentData.members) {
                    currentData.members = {};
                  }

                  // Only increment count if user wasn't already a member
                  if (!currentData.members[currentUserId]) {
                    currentData.memberCount =
                      (currentData.memberCount || 0) + 1;
                    currentData.members[currentUserId] = true;
                  }

                  return currentData;
                }).catch((error) => {
                  console.error("Error updating member count:", error);
                });
              } else {
                // User is already in the participants list, just update their entry
                // Get the first participant key (should only be one)
                snapshot.forEach((childSnapshot) => {
                  newParticipantRef = childSnapshot.ref;
                  // Update the joinedAt timestamp
                  update(newParticipantRef, {
                    joinedAt: serverTimestamp(),
                  });
                  return true; // Break the forEach loop after the first item
                });
              }

              if (window.beforeUnloadHandler) {
                window.removeEventListener(
                  "beforeunload",
                  window.beforeUnloadHandler
                );
              }

              // Create a new handler function
              window.beforeUnloadHandler = function () {
                if (newParticipantRef) {
                  // Remove from participants list
                  remove(newParticipantRef);

                  // Decrement the room's memberCount
                  const roomDataRef = ref(db, `rooms/${roomId}`);
                  runTransaction(roomDataRef, (currentData) => {
                    if (currentData === null) return null;

                    // Remove user from members
                    if (
                      currentData.members &&
                      currentData.members[currentUserId]
                    ) {
                      delete currentData.members[currentUserId];
                      currentData.memberCount = Math.max(
                        1,
                        (currentData.memberCount || 1) - 1
                      );
                    }

                    return currentData;
                  }).catch((error) => {
                    console.error(
                      "Error updating member count on disconnect:",
                      error
                    );
                  });
                }
              };

              // Add the listener
              window.addEventListener(
                "beforeunload",
                window.beforeUnloadHandler
              );
            })
            .catch((error) => {
              console.error("Error checking for existing participant:", error);
            });

          const connectedRef = ref(db, ".info/connected");
          onValue(connectedRef, (snap) => {
            if (snap.val() === true) {
              // We're connected
              const roomMemberRef = ref(
                db,
                `rooms/${roomId}/members/${currentUserId}`
              );

              // Cancel any previous onDisconnect operations
              onDisconnect(roomMemberRef)
                .cancel()
                .then(() => {
                  // Set up new onDisconnect handler
                  onDisconnect(roomMemberRef).remove();

                  // Make sure we're in the members list
                  set(roomMemberRef, true);
                });

              // Set up transaction for member count separately
              const roomDataRef = ref(db, `rooms/${roomId}`);
              onDisconnect(roomDataRef).transaction((currentData) => {
                if (currentData === null) return null;

                // Only decrement if this user is a member
                if (currentData.members && currentData.members[currentUserId]) {
                  delete currentData.members[currentUserId];
                  currentData.memberCount = Math.max(
                    1,
                    (currentData.memberCount || 1) - 1
                  );
                }

                return currentData;
              });
            }
          });
        })
        .catch((error) => {
          console.error("Error joining room:", error);
          alert("Error joining room. Please try again.");
        });
    }

    // Add this function to your JavaScript
    function showToast(message, type = "default") {
      // Create toast container if it doesn't exist
      let toastContainer = document.querySelector(".toast-container");
      if (!toastContainer) {
        toastContainer = document.createElement("div");
        toastContainer.className = "toast-container";
        document.body.appendChild(toastContainer);
      }

      // Create toast element
      const toast = document.createElement("div");
      toast.className = `toast ${type}`;
      toast.textContent = message;

      // Add to container
      toastContainer.appendChild(toast);

      // Remove after animation completes
      setTimeout(() => {
        toast.remove();
      }, 3000);
    }

    // Usage examples:
    // showToast("Room created successfully!", "success");
    // showToast("Failed to join room", "error");
    // showToast("This room is currently full", "warning");

    // Function to load messages for a specific room
    function loadRoomMessages(roomId) {
      const roomMessagesRef = ref(db, `room-messages/${roomId}`);
      const messagesQuery = query(roomMessagesRef, orderByChild("timestamp"));

      onValue(messagesQuery, (snapshot) => {
        const messagesContainer = document.getElementById("messages-container");
        if (!messagesContainer) return;

        messagesContainer.innerHTML = "";

        if (!snapshot.exists()) {
          messagesContainer.innerHTML =
            "<p class='no-messages'>No messages yet. Be the first to say something!</p>";
          return;
        }

        const messages = [];
        snapshot.forEach((childSnapshot) => {
          const message = childSnapshot.val();
          message.key = childSnapshot.key;
          messages.push(message);
        });

        // Sort messages by timestamp
        messages.sort((a, b) => {
          return (a.timestamp || 0) - (b.timestamp || 0);
        });

        // Display messages
        messages.forEach((message) => {
          const messageElement = document.createElement("div");
          messageElement.className = "message";
          messageElement.innerHTML = `
            <span class="username">${message.username || "Anonymous"}:</span>
            <span class="text">${message.text}</span>
          `;
          messagesContainer.appendChild(messageElement);
        });

        // Scroll to bottom
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
      });
    }

    // Update the sendMessage function in app.js to work with rooms
    // We need to override the existing function
    if (window.ChatApp && window.ChatApp.sendMessage) {
      const originalSendMessage = window.ChatApp.sendMessage;

      window.ChatApp.sendMessage = async function () {
        const messageInput = document.getElementById("message-input");
        const usernameInput = document.getElementById("username-input");

        if (!messageInput || !usernameInput) {
          console.error("Message or username input not found");
          return;
        }

        const messageText = messageInput.value.trim();
        const username = usernameInput.value.trim() || "Anonymous";

        if (messageText === "") return;

        // Clear input field
        messageInput.value = "";
        document.getElementById("char-count").textContent = "0";

        // Create user message object
        const userMessage = {
          text: messageText,
          username: username,
          timestamp: serverTimestamp(),
          userId: currentUserId,
          isAI: false,
        };

        try {
          // If in a room, send to room-messages
          if (currentRoomId) {
            const roomMessagesRef = ref(db, `room-messages/${currentRoomId}`);
            const newMessageRef = push(roomMessagesRef);
            await set(newMessageRef, userMessage);

            // Create 3D chat bubble for room message
            if (window.ChatApp.createChatBubble) {
              window.ChatApp.createChatBubble(messageText);
            }

            // If message starts with "/", trigger AI response in the room
            if (messageText.startsWith("/")) {
              console.log("AI TRIGGERED");
              const aiPrompt = messageText.substring(1).trim();

              if (window.ChatApp.generateAIResponseWithContext) {
                console.log("AI service available");
              } else {
                console.error("AI service not available");
              }

              if (window.ChatApp.generateAIResponseWithContext) {
                const aiResponseText =
                  await window.ChatApp.generateAIResponseWithContext(aiPrompt);

                const aiMessage = {
                  text: aiResponseText,
                  username: "🤖 Your Friend",
                  timestamp: serverTimestamp(),
                  isAI: true,
                };

                // Add AI message to the room
                const aiMessageRef = push(roomMessagesRef);
                await set(aiMessageRef, aiMessage);

                // Create AI chat bubble
                if (window.ChatApp.createChatBubble) {
                  window.ChatApp.createChatBubble(aiResponseText, true);
                }
              }
            }
          } else {
            // If not in a room, use the global messages path
            const messagesRef = ref(db, "messages");
            const newMessageRef = push(messagesRef);
            await set(newMessageRef, userMessage);

            // Create 3D chat bubble for global message
            if (window.ChatApp.createChatBubble) {
              window.ChatApp.createChatBubble(messageText);
            }

            // If message starts with "/", trigger AI response in global chat
            if (messageText.startsWith("/")) {
              console.log(
                "AI TRIGGERED with prompt:",
                messageText.substring(1).trim()
              );
              const aiPrompt = messageText.substring(1).trim();

              if (window.ChatApp.generateAIResponseWithContext) {
                const aiResponseText =
                  await window.ChatApp.generateAIResponseWithContext(aiPrompt);

                const aiMessage = {
                  text: aiResponseText,
                  username: "🤖 Your Friend",
                  timestamp: serverTimestamp(),
                  isAI: true,
                };

                // Add AI message to global chat
                const aiMessageRef = push(messagesRef);
                await set(aiMessageRef, aiMessage);

                // Create AI chat bubble
                if (window.ChatApp.createChatBubble) {
                  window.ChatApp.createChatBubble(aiResponseText, true);
                }
              }
            }
          }
        } catch (error) {
          console.error("Error sending message:", error);
        }
      };
    }

    // Helper function to generate a unique user ID
    function generateUserId() {
      // Check if we already have a user ID in localStorage
      let userId = localStorage.getItem("userId");
      if (!userId) {
        // Generate a new ID if none exists
        userId = "user_" + Math.random().toString(36).substring(2, 15);
        localStorage.setItem("userId", userId);
      }
      return userId;
    }

    // Handle Enter key in password input
    roomPasswordInput.addEventListener("keydown", function (e) {
      if (e.key === "Enter") {
        e.preventDefault();
        confirmJoinPrivateRoom();
      }
    });

    // Initialize by loading global messages first
    loadGlobalMessages();
  } catch (error) {
    console.error("Error initializing Firebase for rooms:", error);
  }
});
